#************************************************************************************
#                                                                                                  
#                                 Ex. LCD_I2C.py                                       
#                                                                                                  
#************************************************************************************
#                                                                                                  
#  Board MCU   : RASPBERRY PI 3 MODEL B                                                         
#  Editor-IDE  : Geany 1.29                                                                      
#  Target Board Interface : ET-IOT HAT + LCD 16x2 I2C (I2C-5V)                                                      
#                                                                                                                                                                                                                                             
#************************************************************************************

import smbus
import time

#---Open I2C interface---
bus = smbus.SMBus(1) # use Port i2c-1 


#---Define some device parameters ---
#I2C_ADDR  = 0x3F # LCD I2C device address (PCF8574A)
I2C_ADDR  = 0x27 # LCD I2C device address (PCF8574)

#---Define some device constants---
cmm =  0  #sent command 
data = 1  #Sent data

BL_ON  = 0x08  
BL_OFF = 0x00 
LCD_BL = 0x08


#---------------------------------------
#           Function-Delay
#---------------------------------------
# dy = value delay unit (ms)
#---------------------------------------
def delay_ms(dy):
	time.sleep(dy/1000.0)


#************************ LCD FUNCTION ****************************

#--------------------------------------------------------
#                  Function-Initial LCD
# -------------------------------------------------------  
def init_LCD():
	# Initialise display
	lcd_write(0x33,cmm) # 110011 Initialise
	lcd_write(0x32,cmm) # 110010 Initialise
	lcd_write(0x06,cmm) # 000110 Cursor move direction
	lcd_write(0x0C,cmm) # 001100 Display On,Cursor Off, Blink Off
	lcd_write(0x28,cmm) # 101000 Data length, number of lines, font size
	lcd_write(0x01,cmm) # 000001 Clear display
   
	time.sleep(0.005)   # delay 5 ms
	lcd_BackLight(BL_ON)
  

#-----------------------------------------------------
#           Function-LCD Write Cmm or Data 
#-----------------------------------------------------
# Send byte to data pins
# byte = data
# mode = Sent Command=0=cmm or Sent Data=1=data  
#-----------------------------------------------------
def lcd_write(byte,mode):

	Byte_High = (byte & 0xF0)|LCD_BL| mode       # Mark Data 4 Bit High and Control BL(bit3),En=0(bit2),R/W=0(bit1),RS(bit0)
	Byte_Low  = ((byte<<4) & 0xF0)|LCD_BL| mode  # Mark data 4 Bit Low and Control  BL(bit3),En=0(bit2),R/W=0(bit1),RS(bit0)

	# Sent Data 4 bit High 
	bus.write_byte(I2C_ADDR,Byte_High)    # Write data 1 Byte(data of 4bitHigh and 4Bit Low = Control)
	lcd_enable(Byte_High)                 # LCD'Enable' pin
	
	# Sent Data 4 bit Low
	bus.write_byte(I2C_ADDR,Byte_Low)     # Write data 1 Byte(data of 4bit Low and 4Bit Low = Control)
	lcd_enable(Byte_Low)                  # LCD'Enable' pin


#---------------------------------------------------
#           Function-Toggle LCD Pin Enable
#---------------------------------------------------
# byt = data & Control for LCD 1 Byte
#---------------------------------------------------
def lcd_enable(byt):
	bus.write_byte(I2C_ADDR,(byt & 0xFB))     # Mark EN=0(bit2) and write to LCD
	time.sleep(0.0005)                        # delay 500 us
	bus.write_byte(I2C_ADDR,(byt | 0x04))     # Mark EN=1(bit2) and write to LCD
	time.sleep(0.0005)                        # delay 500 us
	bus.write_byte(I2C_ADDR,(byt & 0xFB))     # Mark EN=0(Bit2) and write to LCD


#-------------------------------------------------
#            Function-ON/OFF LCD Back Light
#-------------------------------------------------
# ONF = BL_ON:0x08  or BL_OFF:0x00
# 
#-------------------------------------------------
def lcd_BackLight(ONF):
	global LCD_BL
	LCD_BL = ONF


#-------------------------------------------------
#            Function-Set Cursor 
#-------------------------------------------------
# colum = 0-15
# line  = 1 or 2
#-------------------------------------------------
def lcd_cursor(colum,line):
	if  line==0x01: 
		line = 0x80 | colum
		lcd_write(line,cmm)  	# Sent Cmm Set Ram Address Line1(0x00)
	if  line==0x02:
		line = 0xC0 | colum 
		lcd_write(line,cmm)  	# Sent Cmm Set Ram Address Line2(0x40)

#--------------------------------------------------------
#                  Function-Clear LCD
# ------------------------------------------------------- 
def lcd_clear():
	lcd_write(0x01,cmm)			# 000001 Clear LCD display 
	
	
#------------------------------------------------
#             Function-Sent String 
#------------------------------------------------
# message = string
#------------------------------------------------   
def lcd_string(message):
	for i in range(len(message)):          # Loop Print message
		lcd_write(ord(message[i]),data)    # Return ASCII Code and Write data to LCD


#********************** END-LCD **********************

print ("Test LCD 16x2 I2C")
print ("Press CTRL-C to exit")

init_LCD()   					# Initialise LCD

try:
	while True:
		lcd_cursor(3,1)				# Set Cursor Colum4 Line1
		lcd_string("ET-IOT HAT")
		lcd_cursor(0,2)				# Set Cursor Colum0 ,Line2
		lcd_string("TEST LCD16x2 I2C")
		delay_ms(2000)				# 2 second delay
		lcd_clear() 		        # Clear LCD display

		lcd_cursor(0,1)
		lcd_string("Hello World")
		lcd_cursor(0,2)
		lcd_string("Raspberry Pi")
		delay_ms(2000)
		lcd_clear()

except KeyboardInterrupt:
	print("End of Program")
  





